"use client";
import { getSelectedLanguage, loadLanguages } from "@/utils/serverUtils";
import { Icon } from "@iconify/react";
import Cookies from "js-cookie";
import { usePathname, useRouter } from "next/navigation";
import { useEffect, useRef, useState } from "react";

const LanguageDropdown = ({ authpage = false }) => {
  const [languageDropdown, setLanguageDropdown] = useState(false);
  const [languages, setLanguages] = useState([]);
  const [selectedLang, setSelectedLang] = useState(null);

  const dropdownRef = useRef(null);
  const pathname = usePathname();
  const router = useRouter();

  useEffect(() => {
    const getLanguages = async () => {
      try {
        const languages = await loadLanguages();
        setLanguages(languages);

        // Get current locale
        const locale = Cookies.get("NEXT_LOCALE") || "en";
        const selectedLanguage = getSelectedLanguage(languages, locale);

        setSelectedLang(selectedLanguage);
        applyLanguageSettings(selectedLanguage);
      } catch (err) {
        console.error("Language load error:", err);
      }
    };

    getLanguages();
  }, []);

  useEffect(() => {
    const handleClickOutside = (event) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target)) {
        setLanguageDropdown(false);
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    setLanguageDropdown(false);
  }, [pathname]);

  const applyLanguageSettings = (lang) => {
    console.log("Applying language settings for:", lang);
    document.documentElement.dir = lang.is_rtl ? "rtl" : "ltr";
    document.documentElement.lang = lang.locale;
    document.documentElement.setAttribute("data-language", lang.locale);

    console.log("is rtl", lang.is_rtl);
    console.log("lang", lang.locale);
  };

  const handleSelect = (lang) => {
    setSelectedLang(lang);
    setLanguageDropdown(false);

    // Apply visual settings immediately
    applyLanguageSettings(lang);

    // Update cookie
    Cookies.set("NEXT_LOCALE", lang.locale, { expires: 365 });

    // Dispatch event for TranslationProvider
    window.dispatchEvent(
      new CustomEvent("languageChange", {
        detail: { language: lang },
      })
    );

    // Refresh to reload server components
    router.refresh();
  };

  if (!selectedLang) return null;

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        className={`h-[35px] sm:h-[40px] flex items-center gap-[10px] cursor-pointer px-[16px] rounded-[12px] ${
          authpage ? "bg-[#DBF6E6]" : "bg-white"
        }`}
        onClick={() => setLanguageDropdown((prev) => !prev)}
      >
        <Icon
          icon="lucide:globe"
          className="text-merchant-text"
          width="18"
          height="18"
        />
        <span className="text-merchant-text text-[14px] font-semibold">
          {selectedLang.name}
        </span>
      </button>

      <div
        className={`absolute ${
          authpage ? "rtl:right-0 ltr:left-0" : "ltr:right-0 rtl:left-0"
        } mt-[10px] w-[153px] bg-white border-2 border-[rgba(116,69,255,0.16)]
        rounded-[16px] shadow-lg z-10 transform transition-all duration-300 origin-top
        ${
          languageDropdown
            ? "opacity-100 scale-y-100"
            : "opacity-0 scale-y-0 pointer-events-none"
        }`}
      >
        <ul>
          {languages.map((lang) => (
            <li key={lang.code}>
              <button
                onClick={() => handleSelect(lang)}
                className={`w-full px-[16px] py-[10px] ltr:text-left rtl:text-right text-[14px] font-semibold transition-colors
                  ${
                    selectedLang.locale === lang.locale
                      ? "text-merchant-primary"
                      : "text-merchant-text hover:text-merchant-primary"
                  }`}
              >
                {lang.name}
              </button>
            </li>
          ))}
        </ul>
      </div>
    </div>
  );
};

export default LanguageDropdown;
