"use client";

import dotBg from "@/assets/dashboard/background/dot-bg.svg";
import successIcon from "@/assets/dashboard/icon/success-image.svg";
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectLabel,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useSettings } from "@/context/settingsContext";
import { useT } from "@/context/TranslationContext";
import ApiPath from "@/network/api/apiPath";
import NetworkService from "@/network/service/networkService";
import { dynamicDecimals, getSettingValue } from "@/utils/utils";
import { Icon } from "@iconify/react";
import Image from "next/image";
import { useEffect, useState } from "react";
import { toast } from "react-toastify";

const Exchange = () => {
  const network = new NetworkService();
  const [wallets, setWallets] = useState([]);
  const [walletsLoading, setWalletsLoading] = useState(false);
  const [fromWallet, setFromWallet] = useState("");
  const [toWallet, setToWallet] = useState("");
  const [selectAmount, setSelectAmount] = useState("");
  const [configData, setConfigData] = useState([]);
  const [calculatedCharge, setCalculatedCharge] = useState(0);
  const [totalAmount, setTotalAmount] = useState(0);
  const [exchangeConvertedAmount, setExchangeConvertedAmount] = useState(0);
  const [exchangeResponse, setExchangeResponse] = useState(null);
  const [loading, setLoading] = useState(false);
  const [step, setStep] = useState(1);
  const { settings } = useSettings();
  const siteCurrencyCode = getSettingValue(settings, "site_currency");
  const siteCurrencyDecimals = getSettingValue(
    settings,
    "site_currency_decimals"
  );
  const t = useT();

  const steps = [
    { num: 1, label: t("title.amount"), sublabel: t("title.step1") },
    { num: 2, label: t("title.review"), sublabel: t("title.step2") },
    { num: 3, label: t("title.success"), sublabel: t("title.step3") },
  ];

  // Fetch wallets
  const walletsData = async () => {
    try {
      setWalletsLoading(true);
      const res = await network.get(ApiPath.wallets, {
        exchange: 1,
      });
      if (res.status === "completed") {
        setWallets(res.data.data.wallets);
      }
    } finally {
      setWalletsLoading(false);
    }
  };

  // selected wallets data
  const selectedFromWalletData = fromWallet
    ? wallets.find((w) =>
        fromWallet === "default"
          ? w.is_default
          : String(w.id) === String(fromWallet)
      )
    : null;
  const selectedToWalletData = toWallet
    ? wallets.find((w) =>
        toWallet === "default"
          ? w.is_default
          : String(w.id) === String(toWallet)
      )
    : null;

  // Filter wallets for select lists
  const availableFromWallets = wallets.filter(
    (wallet) =>
      wallet.id !== toWallet && !(wallet.is_default && toWallet === "default")
  );

  const availableToWallets = wallets.filter(
    (wallet) =>
      wallet.id !== fromWallet &&
      !(wallet.is_default && fromWallet === "default")
  );

  //fetch exchange config
  const fetchExchangeConfig = async () => {
    try {
      const res = await network.get(ApiPath.exchangeConfig);
      setConfigData(res.data.data);
    } finally {
    }
  };

  const reviewCalculate = async () => {
    const baseAmount = parseFloat(selectAmount);
    let chargeValue = 0;

    const chargeType = configData.settings.charge_type;
    const chargeAmount = parseFloat(configData.settings.charge);

    if (chargeType?.toLowerCase() === "percentage") {
      chargeValue = (baseAmount * chargeAmount) / 100;
      const total = baseAmount + chargeValue;
      setCalculatedCharge(chargeValue);
      setTotalAmount(total);
    } else if (chargeType?.toLowerCase() === "fixed") {
      await convertFixedAmount(chargeAmount, baseAmount);
    }
  };

  // if fixed amount
  const convertFixedAmount = async (charge, amount) => {
    try {
      const res = await network.globalGet(
        ApiPath.currencyConvert(charge, selectedFromWalletData?.code)
      );

      if (res.status === "completed") {
        const convertedCharge = Number(res.data.data.converted_amount);
        setCalculatedCharge(convertedCharge);
        const total = amount + convertedCharge;
        setTotalAmount(total);
      }
    } catch (e) {
      console.error(e.response?.data || e.message);
    }
  };

  const getExchangeRateConverter = async () => {
    try {
      const res = await network.globalGet(
        ApiPath.exchangeConvert(
          selectAmount,
          selectedToWalletData?.code,
          selectedFromWalletData?.code
        )
      );

      if (res.status === "completed") {
        setExchangeConvertedAmount(res.data.data);
      }
    } finally {
    }
  };

  const handleExchangeSubmit = async () => {
    setLoading(true);
    try {
      const requestBody = {
        from_wallet: selectedFromWalletData?.is_default
          ? "default"
          : selectedFromWalletData?.id,
        to_wallet: selectedToWalletData?.is_default
          ? "default"
          : selectedToWalletData?.id,
        amount: selectAmount,
      };
      const res = await network.post(ApiPath.exchange, requestBody);
      if (res.status === "completed") {
        toast.success(res.data.message);
        setExchangeResponse(res.data.data);
        setStep(3);
      }
    } finally {
      setLoading(false);
    }
  };

  const validateStep1 = () => {
    if (!fromWallet) {
      toast.error("Please select a from wallet");
      return false;
    }
    if (!toWallet) {
      toast.error("Please select a to wallet");
      return false;
    }
    if (!selectAmount || parseFloat(selectAmount) <= 0) {
      toast.error("Please enter a valid amount");
      return false;
    }
    if (
      selectedFromWalletData?.exchange_limit?.min &&
      selectedFromWalletData?.exchange_limit?.max
    ) {
      const numAmount = parseFloat(selectAmount);
      const numMin = parseFloat(selectedFromWalletData?.exchange_limit?.min);
      const numMax = parseFloat(selectedFromWalletData?.exchange_limit?.max);

      if (numAmount < numMin) {
        toast.error(
          `Amount must be at least ${numMin} ${selectedFromWalletData?.code}`
        );
        return false;
      }

      if (numAmount > numMax) {
        toast.error(
          `Amount cannot exceed ${numMax} ${selectedFromWalletData?.code}`
        );
        return false;
      }
    }
    return true;
  };

  // Step navigation
  const nextStep = () => {
    if (step === 1) {
      if (!validateStep1()) {
        return;
      }
    }
    setStep((prev) => Math.min(prev + 1, 2));
  };
  const prevStep = () => setStep((prev) => Math.max(prev - 1, 1));
  const resetSteps = () => {
    setStep(1);
    setFromWallet("");
    setToWallet("");
    setSelectAmount("");
    setCalculatedCharge(0);
    setTotalAmount(0);
    setExchangeConvertedAmount(0);
    setExchangeResponse(null);
  };

  // Effects
  useEffect(() => {
    walletsData();
    fetchExchangeConfig();
  }, []);

  useEffect(() => {
    if (selectAmount && configData?.settings && fromWallet) {
      reviewCalculate();
    }
  }, [selectAmount, configData, fromWallet]);

  useEffect(() => {
    if (selectAmount && selectedFromWalletData && selectedToWalletData) {
      getExchangeRateConverter();
    }
  }, [selectAmount, selectedFromWalletData, selectedToWalletData]);

  return (
    <div className="w-full border border-[rgba(26,32,44,0.10)] rounded-[8px] p-[16px] sm:p-[30px] xl:p-[60px]">
      <div className="w-full max-w-[970px]">
        <div className="flex flex-col lg:flex-row gap-12">
          {/* Steps */}
          <div className="w-full lg:w-[130px] xl:w-[160px] 2xl:w-[250px] 3xl:w-[350px] flex-shrink-0">
            <div className="relative flex justify-between items-center lg:block">
              {steps.map((singleStep, index) => (
                <div key={singleStep.num} className="relative w-full">
                  {index < steps.length - 1 && (
                    <div
                      className={`absolute rtl:right-1/2 ltr:left-1/2 lg:rtl:right-5 lg:ltr:left-5 top-[22px] lg:top-10 w-full lg:w-0.5 h-0.5 lg:h-[100px] transition-colors duration-300 ${
                        step > singleStep.num
                          ? "bg-merchant-primary"
                          : "bg-gray-300"
                      }`}
                    ></div>
                  )}

                  <div className="flex flex-col lg:flex-row items-center gap-4 mb-0 lg:mb-[65px] text-center lg:text-left">
                    <div
                      className={`w-10 h-10 flex-shrink-0 rounded-full flex items-center justify-center transition-all duration-300 relative after:absolute after:top-1/2 after:left-1/2 after:-translate-1/2 after:rounded-full after:content-[""] after:w-[41px] after:h-[41px] after:bg-transparent after:border-4 after:border-white ${
                        step === singleStep.num
                          ? "bg-merchant-primary"
                          : step > singleStep.num
                          ? "bg-merchant-primary"
                          : "bg-gray-300"
                      }`}
                    >
                      {step >= singleStep.num ? (
                        <svg
                          className="w-6 h-6 text-white"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M5 13l4 4L19 7"
                          />
                        </svg>
                      ) : (
                        <div className="w-[15px] h-[15px] rounded-full bg-white"></div>
                      )}
                    </div>

                    <div>
                      <div
                        className={`font-semibold ${
                          step >= singleStep.num
                            ? "text-merchant-text"
                            : "text-merchant-paragraph"
                        }`}
                      >
                        {singleStep.label}
                      </div>
                      <div className="text-sm text-gray-500">
                        {singleStep.sublabel}
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Content */}
          <div className="flex-1 rtl:border-r-0 ltr:border-l-0 lg:rtl:border-r lg:ltr:border-l border-l-[rgba(26,32,44,0.10)] rtl:pr-0 ltr:pl-0 lg:rtl:pr-[30px] lg:ltr:pl-[30px] xl:rtl:pr-[80px] xl:ltr:pl-[80px]">
            {step === 1 && (
              <div className="step-1-content">
                <div className="grid grid-cols-12 gap-5">
                  <div className="col-span-12">
                    <div className="relative">
                      <Select
                        value={fromWallet}
                        onValueChange={(value) => setFromWallet(value)}
                        disabled={walletsLoading}
                      >
                        <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[50%] -translate-y-1/2">
                          <Icon
                            icon="basil:wallet-solid"
                            width="24"
                            height="24"
                            className="w-[18px] h-[18px] text-[#9E9A9F]"
                          />
                        </div>

                        <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[-12px]">
                          <span className="text-[13px] text-[rgba(26,32,44,0.60)] font-semibold bg-white px-[10px]">
                            {t("title.fromWallet")}{" "}
                            <span className="text-merchant-error">*</span>
                          </span>
                        </div>

                        <SelectTrigger className="w-full !h-[52px] border-2 border-merchant-input-border rounded-[8px] text-merchant-text ps-[42px]">
                          <SelectValue
                            placeholder={
                              walletsLoading
                                ? t("placeholder.loadingWallets")
                                : t("placeholder.selectFromWallet")
                            }
                          />
                        </SelectTrigger>

                        <SelectContent className="rounded-[8px]">
                          <SelectGroup>
                            <SelectLabel>
                              {t("title.availableWallets")}
                            </SelectLabel>

                            {availableFromWallets.map((wallet) => (
                              <SelectItem
                                key={wallet.id}
                                value={
                                  wallet.is_default ? "default" : wallet.id
                                }
                              >
                                {wallet.name} ({wallet.formatted_balance}{" "}
                                {wallet.code})
                              </SelectItem>
                            ))}
                          </SelectGroup>
                        </SelectContent>
                      </Select>
                    </div>
                  </div>
                  <div className="col-span-12">
                    <div className="relative">
                      <Select
                        value={toWallet}
                        onValueChange={(value) => setToWallet(value)}
                        disabled={walletsLoading}
                      >
                        <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[50%] -translate-y-1/2">
                          <Icon
                            icon="basil:wallet-solid"
                            width="24"
                            height="24"
                            className="w-[18px] h-[18px] text-[#9E9A9F]"
                          />
                        </div>

                        <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[-12px]">
                          <span className="text-[13px] text-[rgba(26,32,44,0.60)] font-semibold bg-white px-[10px]">
                            {t("title.toWallet")}{" "}
                            <span className="text-merchant-error">*</span>
                          </span>
                        </div>

                        <SelectTrigger className="w-full !h-[52px] border-2 border-merchant-input-border rounded-[8px] text-merchant-text ps-[42px]">
                          <SelectValue
                            placeholder={
                              walletsLoading
                                ? t("placeholder.loadingWallets")
                                : t("placeholder.selectToWallet")
                            }
                          />
                        </SelectTrigger>

                        <SelectContent className="rounded-[8px]">
                          <SelectGroup>
                            <SelectLabel>
                              {t("title.availableWallets")}
                            </SelectLabel>

                            {availableToWallets.map((wallet) => (
                              <SelectItem
                                key={wallet.id}
                                value={
                                  wallet.is_default ? "default" : wallet.id
                                }
                              >
                                {wallet.name} ({wallet.formatted_balance}{" "}
                                {wallet.code})
                              </SelectItem>
                            ))}
                          </SelectGroup>
                        </SelectContent>
                      </Select>
                    </div>
                  </div>
                  <div className="col-span-12">
                    <div className="relative">
                      <input
                        type="text"
                        id="amount"
                        className={`user-input user-input-left peer`}
                        placeholder=" "
                        value={selectAmount}
                        onChange={(e) => setSelectAmount(e.target.value)}
                      />
                      <label
                        htmlFor="amount"
                        className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[30px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                      >
                        {t("title.amount")}{" "}
                        <span className="text-merchant-error">*</span>
                      </label>
                      <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[50%] -translate-y-1/2 z-20">
                        <Icon
                          icon="lets-icons:money-fill"
                          width="24"
                          height="24"
                          className="text-[#9A9DA2] h-[20px] w-[20px]"
                        />
                      </div>
                    </div>
                    {selectedFromWalletData && (
                      <p className="text-red-500 text-[13px] font-medium mt-1">
                        {t("label.minimum")}{" "}
                        {selectedFromWalletData?.exchange_limit?.min}{" "}
                        {selectedFromWalletData?.code} | {t("label.maximum")}{" "}
                        {selectedFromWalletData?.exchange_limit?.max}{" "}
                        {selectedFromWalletData?.code}
                      </p>
                    )}
                  </div>
                </div>

                <div className="mt-[40px]">
                  <button
                    onClick={nextStep}
                    className="group primary-button w-full"
                  >
                    <span className="primary-button-hover-effect"></span>
                    <span className="primary-button-text">
                      {t("buttonText.next")}
                    </span>
                  </button>
                </div>
              </div>
            )}

            {step === 2 && (
              <div className="step-2-content">
                <div className="details">
                  <h5 className="text-[18px] font-bold text-merchant-text mb-[16px]">
                    {t("title.reviewDetails")}
                  </h5>
                  <div className="border border-[rgba(26,32,44,0.10)] rounded-[8px]">
                    <div className="flex flex-col sm:flex-row gap-1.5 justify-between items-start sm:items-center border-b border-[rgba(26,32,44,0.10)] px-[12px] py-[16px]">
                      <h6 className="text-[14px] font-semibold text-merchant-paragraph">
                        {t("title.amount")}
                      </h6>
                      <p className="text-[14px] font-semibold text-merchant-primary">
                        {Number(selectAmount).toFixed(
                          dynamicDecimals({
                            currencyCode: selectedFromWalletData?.code,
                            siteCurrencyCode: siteCurrencyCode,
                            siteCurrencyDecimals: siteCurrencyDecimals,
                            isCrypto: selectedFromWalletData?.is_crypto,
                          })
                        )}{" "}
                        {selectedFromWalletData?.code}
                      </p>
                    </div>
                    <div className="flex flex-col sm:flex-row gap-1.5 justify-between items-start sm:items-center border-b border-[rgba(26,32,44,0.10)] px-[12px] py-[16px]">
                      <h6 className="text-[14px] font-semibold text-merchant-paragraph">
                        {t("title.fromWallet")}
                      </h6>
                      <p className="text-[14px] font-semibold text-merchant-text">
                        {selectedFromWalletData?.name}
                      </p>
                    </div>
                    <div className="flex flex-col sm:flex-row gap-1.5 justify-between items-start sm:items-center border-b border-[rgba(26,32,44,0.10)] px-[12px] py-[16px]">
                      <h6 className="text-[14px] font-semibold text-merchant-paragraph">
                        {t("title.toWallet")}
                      </h6>
                      <p className="text-[14px] font-semibold text-merchant-text">
                        {selectedToWalletData?.name}
                      </p>
                    </div>
                    <div className="flex flex-col sm:flex-row gap-1.5 justify-between items-start sm:items-center border-b border-[rgba(26,32,44,0.10)] px-[12px] py-[16px]">
                      <h6 className="text-[14px] font-semibold text-merchant-paragraph">
                        {t("title.exchangeRate")}
                      </h6>
                      <p className="text-[14px] font-semibold text-merchant-text">
                        1 {selectedFromWalletData?.code} ={" "}
                        {Number(exchangeConvertedAmount?.rate).toFixed(
                          dynamicDecimals({
                            currencyCode: selectedToWalletData?.code,
                            siteCurrencyCode: siteCurrencyCode,
                            siteCurrencyDecimals: siteCurrencyDecimals,
                            isCrypto: selectedToWalletData?.is_crypto,
                          })
                        )}{" "}
                        {selectedToWalletData?.code}
                      </p>
                    </div>
                    <div className="flex flex-col sm:flex-row gap-1.5 justify-between items-start sm:items-center border-b border-[rgba(26,32,44,0.10)] px-[12px] py-[16px]">
                      <h6 className="text-[14px] font-semibold text-merchant-paragraph">
                        {t("title.exchangeAmount")}
                      </h6>
                      <p className="text-[14px] font-semibold text-merchant-text">
                        {Number(
                          exchangeConvertedAmount?.converted_amount
                        ).toFixed(
                          dynamicDecimals({
                            currencyCode: selectedToWalletData?.code,
                            siteCurrencyCode: siteCurrencyCode,
                            siteCurrencyDecimals: siteCurrencyDecimals,
                            isCrypto: selectedToWalletData?.is_crypto,
                          })
                        )}{" "}
                        {selectedToWalletData?.code}
                      </p>
                    </div>
                    <div className="flex flex-col sm:flex-row gap-1.5 justify-between items-start sm:items-center border-b border-[rgba(26,32,44,0.10)] px-[12px] py-[16px]">
                      <h6 className="text-[14px] font-semibold text-merchant-paragraph">
                        {t("title.charge")}
                      </h6>
                      <p className="text-[14px] font-semibold text-merchant-error">
                        {Number(calculatedCharge).toFixed(
                          dynamicDecimals({
                            currencyCode: selectedFromWalletData?.code,
                            siteCurrencyCode: siteCurrencyCode,
                            siteCurrencyDecimals: siteCurrencyDecimals,
                            isCrypto: selectedFromWalletData?.is_crypto,
                          })
                        )}{" "}
                        {selectedFromWalletData?.code}
                      </p>
                    </div>
                    <div className="flex flex-col sm:flex-row gap-1.5 justify-between items-start sm:items-center px-[12px] py-[16px]">
                      <h6 className="text-[14px] font-semibold text-merchant-paragraph">
                        {t("title.totalAmount")}
                      </h6>
                      <p className="text-[16px] font-semibold text-merchant-text">
                        {Number(totalAmount).toFixed(
                          dynamicDecimals({
                            currencyCode: selectedFromWalletData?.code,
                            siteCurrencyCode: siteCurrencyCode,
                            siteCurrencyDecimals: siteCurrencyDecimals,
                            isCrypto: selectedFromWalletData?.is_crypto,
                          })
                        )}{" "}
                        {selectedFromWalletData?.code}
                      </p>
                    </div>
                  </div>
                </div>

                <div className="flex sm:flex-row flex-col-reverse gap-4 mt-[40px]">
                  <button
                    onClick={prevStep}
                    className="group primary-button secondary-color-btn w-full"
                  >
                    <span className="primary-button-hover-effect secondary-button-hover-effect"></span>
                    <span className="primary-button-text">
                      {t("buttonText.backButton")}
                    </span>
                  </button>
                  <button
                    onClick={handleExchangeSubmit}
                    className="group primary-button w-full"
                    disabled={loading}
                  >
                    <span className="primary-button-hover-effect"></span>
                    <span className="primary-button-text">
                      {loading
                        ? t("buttonText.exchanging")
                        : t("buttonText.exchange")}
                    </span>
                  </button>
                </div>
              </div>
            )}

            {step === 3 && (
              <div className="step-3-content">
                <div className="success-content overflow-hidden">
                  <div className="success-content-up bg-merchant-bg-one rounded-t-[8px] relative overflow-hidden">
                    <div className="absolute top-0 rtl:right-0 ltr:left-0 w-full h-auto">
                      <Image
                        src={dotBg}
                        alt="background"
                        width={100}
                        height={100}
                        className="w-full h-auto object-cover z-[1]"
                      />
                    </div>
                    <div className="relative z-[2] p-[20px] sm:p-[30px] flex flex-col items-center">
                      <div className="w-[80px] h-[80px] mb-[16px]">
                        <Image
                          src={successIcon}
                          alt="success"
                          width={100}
                          height={100}
                          className="w-full h-full object-cover"
                        />
                      </div>
                      <div className="text-center">
                        <h3 className="text-[20px] font-bold text-merchant-text mb-[10px]">
                          {t("title.exchangeSuccess")}
                        </h3>
                        <p className="text-[14px] font-medium text-merchant-paragraph">
                          {t("title.exchangeCompleted")}
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="success-content-separator relative">
                    <div className="w-full border-t-2 border-dashed border-merchant-primary"></div>
                    <div className="w-[22px] h-[22px] bg-white absolute top-1/2 rtl:right-[-12px] ltr:left-[-12px] -translate-y-1/2 rounded-full"></div>
                    <div className="w-[22px] h-[22px] bg-white absolute top-1/2 rtl:left-[-12px] ltr:right-[-12px] -translate-y-1/2 rounded-full"></div>
                  </div>
                  <div className="success-content-down bg-merchant-bg-one rounded-b-[8px] p-[20px] sm:p-[30px]">
                    <div className="bg-white p-[20px] rounded-[8px]">
                      <div className="flex sm:flex-row flex-col justify-start sm:justify-between items-start sm:items-center gap-2 mb-[20px]">
                        <p className="text-[14px] font-semibold text-merchant-paragraph">
                          {t("title.transactionId")}:
                        </p>
                        <p className="text-[14px] font-semibold text-merchant-text">
                          {exchangeResponse?.transaction?.tnx}
                        </p>
                      </div>
                      <div className="flex sm:flex-row flex-col justify-start sm:justify-between items-start sm:items-center gap-2 mb-[20px]">
                        <p className="text-[14px] font-semibold text-merchant-paragraph">
                          {t("title.amount")}
                        </p>
                        <p className="text-[14px] font-semibold text-merchant-text">
                          {exchangeResponse?.transaction?.final_amount}{" "}
                          {selectedToWalletData?.code}
                        </p>
                      </div>
                      <div className="flex sm:flex-row flex-col justify-start sm:justify-between items-start sm:items-center gap-2 mb-[20px]">
                        <p className="text-[14px] font-semibold text-merchant-paragraph">
                          {t("title.fromWallet")}
                        </p>
                        <p className="text-[14px] font-semibold text-merchant-text">
                          {exchangeResponse?.transaction?.pay_currency}
                        </p>
                      </div>
                      <div className="flex sm:flex-row flex-col justify-start sm:justify-between items-start sm:items-center gap-2 mb-[20px]">
                        <p className="text-[14px] font-semibold text-merchant-paragraph">
                          {t("title.toWallet")}
                        </p>
                        <p className="text-[14px] font-semibold text-merchant-text">
                          {exchangeResponse?.transaction?.receive_currency}
                        </p>
                      </div>
                    </div>
                    <div className="mt-[30px]">
                      <button
                        onClick={resetSteps}
                        className="group primary-button w-full"
                      >
                        <span className="primary-button-hover-effect"></span>
                        <span className="primary-button-text">
                          {t("buttonText.exchangeAgain")}
                        </span>
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default Exchange;
