"use client";

import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectLabel,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useT } from "@/context/TranslationContext";
import ApiPath from "@/network/api/apiPath";
import NetworkService from "@/network/service/networkService";
import { Icon } from "@iconify/react";
import Link from "next/link";
import { useRouter } from "next/navigation";
import { useEffect, useState } from "react";
import Flatpickr from "react-flatpickr";
import { toast } from "react-toastify";

const EditInvoice = ({ invoiceId }) => {
  const network = new NetworkService();
  const [items, setItems] = useState([
    { id: 0, name: "", quantity: "", price: "", subtotal: "" },
  ]);
  const [to, setTo] = useState("");
  const [email, setEmail] = useState("");
  const [address, setAddress] = useState("");
  const [currency, setCurrency] = useState("");
  const [issueDate, setIssueDate] = useState("");
  const [published, setPublished] = useState("");
  const [singleInvoiceData, setSingleInvoiceData] = useState({});
  const [wallets, setWallets] = useState([]);
  const [walletsLoading, setWalletsLoading] = useState(false);
  const [loading, setLoading] = useState(false);
  const router = useRouter();
  const [selectedWallet, setSelectedWallet] = useState(null);
  const t = useT();

  // get wallets data
  const walletsData = async () => {
    try {
      setWalletsLoading(true);
      const res = await network.get(ApiPath.wallets);
      if (res.status === "completed") {
        setWallets(res.data.data.wallets);
      }
    } finally {
      setWalletsLoading(false);
    }
  };

  // fetch invoice data
  const fetchSingleInvoiceData = async () => {
    try {
      const res = await network.get(ApiPath.invoiceID(invoiceId));
      if (res.status === "completed") {
        const invoiceData = res.data.data;
        setSingleInvoiceData(invoiceData);
        setTo(invoiceData?.to);
        setEmail(invoiceData?.email);
        setAddress(invoiceData?.address);
        setCurrency(invoiceData?.currency);
        setIssueDate(
          invoiceData?.issue_date ? invoiceData.issue_date.split("T")[0] : ""
        );
        setPublished(invoiceData?.is_published ? "1" : "0");

        if (invoiceData?.items && invoiceData.items.length > 0) {
          const formattedItems = invoiceData.items.map((item, index) => ({
            id: index,
            name: item.name,
            quantity: item.quantity.toString(),
            price: item.unit_price.toString(),
            subtotal: item.subtotal.toString(),
          }));
          setItems(formattedItems);
        }
      }
    } finally {
    }
  };

  // Add new item with incremental ID
  const handleAddItem = () => {
    setItems((prev) => [
      ...prev,
      {
        id: prev.length > 0 ? prev[prev.length - 1].id + 1 : 0,
        name: "",
        quantity: "",
        price: "",
        subtotal: "",
      },
    ]);
  };

  // handle update invoice
  const handleUpdateInvoice = async (e) => {
    e.preventDefault();
    setLoading(true);
    try {
      const requestBody = {
        to: to,
        email: email,
        address: address,
        currency: currency,
        issue_date: issueDate ? new Date(issueDate).toISOString() : null,
        is_published: published,
        _method: "put",
        items: items.map((item) => ({
          name: item.name,
          quantity: item.quantity,
          unit_price: item.price,
        })),
      };

      const res = await network.postFormData(
        ApiPath.invoiceID(singleInvoiceData.id),
        requestBody
      );

      if (res.status === "completed") {
        toast.success("Invoice update successfully!");
        router.push("/dashboard/invoice");
      }
    } finally {
      setLoading(false);
    }
  };

  // Remove item but keep at least one
  const handleRemoveItem = (id) => {
    if (items.length === 1) {
      toast.warning("At least one item must remain!");
      return;
    }
    setItems(items.filter((item) => item.id !== id));
  };

  // Update item fields
  const handleChange = (id, field, value) => {
    setItems((prev) =>
      prev.map((item) =>
        item.id === id
          ? {
              ...item,
              [field]: value,
              subtotal:
                field === "quantity" || field === "price"
                  ? (field === "quantity"
                      ? Number(value)
                      : Number(item.quantity)) *
                    (field === "price" ? Number(value) : Number(item.price))
                  : item.subtotal,
            }
          : item
      )
    );
  };

  useEffect(() => {
    walletsData();
  }, []);

  useEffect(() => {
    fetchSingleInvoiceData();
  }, [invoiceId]);

  useEffect(() => {
    if (wallets.length > 0 && currency) {
      const wallet = wallets.find((w) => w.code === currency);
      setSelectedWallet(wallet || null);
    }
  }, [wallets, currency]);

  return (
    <div>
      <Link
        href="/dashboard/invoice"
        className="text-merchant-text text-[14px] font-bold flex items-center gap-2 hover:underline hover:text-merchant-primary transition-all duration-300 ease-in-out mb-3"
      >
        <Icon
          icon="lucide:arrow-left"
          width="20"
          height="20"
          className="w-[20px] h-[20px]"
        />
        {t("buttonText.backButton")}
      </Link>
      <div className="max-w-[786px] mx-auto">
        <div className="mb-8">
          <h2 className="text-[20px] font-bold text-merchant-text mb-[30px]">
            {t("title.editInvoiceInformation")}
          </h2>
        </div>
        <form onSubmit={handleUpdateInvoice}>
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-7 mb-8">
            <div className="relative">
              <input
                type="text"
                id="to"
                className={`user-input peer`}
                placeholder=" "
                value={to}
                onChange={(e) => setTo(e.target.value)}
              />
              <label
                htmlFor="to"
                className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
              >
                {t("label.invoiceTo")}{" "}
                <span className="text-merchant-error">*</span>
              </label>
            </div>

            <div className="relative">
              <input
                type="email"
                id="email"
                className={`user-input peer`}
                placeholder=" "
                value={email}
                onChange={(e) => setEmail(e.target.value)}
              />
              <label
                htmlFor="email"
                className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
              >
                {t("label.emailAddress")}{" "}
                <span className="text-merchant-error">*</span>
              </label>
            </div>

            <div className="relative">
              <input
                type="text"
                id="address"
                className={`user-input peer`}
                placeholder=" "
                value={address}
                onChange={(e) => setAddress(e.target.value)}
              />
              <label
                htmlFor="address"
                className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
              >
                {t("label.address")}{" "}
                <span className="text-merchant-error">*</span>
              </label>
            </div>

            <div className="relative">
              <Select
                value={selectedWallet?.code || ""}
                onValueChange={(value) => {
                  const wallet = wallets.find((w) => w.code === value);
                  setSelectedWallet(wallet || null);
                  setCurrency(value);
                }}
                disabled={walletsLoading}
              >
                <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[-12px]">
                  <span className="text-[13px] text-[rgba(26,32,44,0.60)] font-semibold bg-white px-[10px]">
                    {t("placeholder.selectWallet")}{" "}
                    <span className="text-merchant-error">*</span>
                  </span>
                </div>
                <SelectTrigger className="w-full !h-[48px] border-2 border-merchant-input-border rounded-[8px] text-merchant-text focus:outline-none focus:shadow-outline bg-white">
                  <SelectValue
                    placeholder={
                      walletsLoading
                        ? t("placeholder.loadingWallets")
                        : t("placeholder.selectWallet")
                    }
                  >
                    {selectedWallet && (
                      <span>
                        {selectedWallet.name} (
                        {selectedWallet.formatted_balance} {selectedWallet.code}
                        )
                      </span>
                    )}
                  </SelectValue>
                </SelectTrigger>

                <SelectContent className="rounded-[16px]">
                  <SelectGroup>
                    <SelectLabel>
                      {t("placeholder.availableWallets")}
                    </SelectLabel>
                    {wallets.map((wallet) => (
                      <SelectItem key={wallet.id} value={wallet.code}>
                        {wallet.name} ({wallet.formatted_balance} {wallet.code})
                      </SelectItem>
                    ))}
                  </SelectGroup>
                </SelectContent>
              </Select>
            </div>

            <div className="relative">
              <label className="user-label absolute rtl:right-[16px] ltr:left-[16px] top-[-8px] !text-[12px]">
                {t("label.issueDate")} <span className="text-red-500">*</span>
              </label>
              <Flatpickr
                className="border-2 border-merchant-input-border rounded-[8px] px-3 h-[48px] text-sm focus:outline-none focus:ring-1 focus:ring-transparent focus:border-merchant-primary w-full bg-white text-gray-900"
                placeholder={t("placeholder.selectDate")}
                options={{
                  dateFormat: "Y-m-d",
                  mode: "single",
                }}
                value={issueDate ? new Date(issueDate) : null}
                onChange={([selectedDate]) => {
                  if (selectedDate) {
                    const yyyy = selectedDate.getFullYear();
                    const mm = String(selectedDate.getMonth() + 1).padStart(
                      2,
                      "0"
                    );
                    const dd = String(selectedDate.getDate()).padStart(2, "0");
                    setIssueDate(`${yyyy}-${mm}-${dd}`);
                  } else {
                    setIssueDate("");
                  }
                }}
              />
            </div>

            <div className="relative">
              <Select
                value={published}
                onValueChange={(value) => setPublished(value)}
              >
                <div className="absolute rtl:right-[16px] ltr:left-[16px] top-[-12px]">
                  <span className="text-[13px] text-[rgba(26,32,44,0.60)] font-semibold bg-white px-[10px]">
                    {t("placeholder.selectStatus")}{" "}
                    <span className="text-merchant-error">*</span>
                  </span>
                </div>
                <SelectTrigger className="w-full !h-[48px] border-2 border-merchant-input-border rounded-[8px] text-merchant-text focus:outline-none focus:shadow-outline bg-white cursor-pointer">
                  <SelectValue placeholder="Select Status">
                    {published === "0"
                      ? "Draft"
                      : published === "1"
                      ? "Published"
                      : null}
                  </SelectValue>
                </SelectTrigger>

                <SelectContent className="rounded-[16px]">
                  <SelectGroup>
                    <SelectLabel>Invoice Status</SelectLabel>
                    <SelectItem value="0">Draft</SelectItem>
                    <SelectItem value="1">Published</SelectItem>
                  </SelectGroup>
                </SelectContent>
              </Select>
            </div>
          </div>

          <div className="flex justify-between items-center mb-6">
            <h2 className="text-[20px] font-bold text-merchant-text">
              {t("title.invoiceItems")}
            </h2>
            <button
              type="button"
              onClick={handleAddItem}
              className="h-[36px] flex items-center gap-[6px] border-2 border-merchant-primary rounded-[8px] px-[10px] text-[14px] font-bold text-merchant-text cursor-pointer transition-all duration-300 ease-in-out hover:bg-merchant-primary hover:text-white hover:border-merchant-primary"
            >
              <Icon icon="lucide:plus" width="20" height="20" />
              {t("buttonText.addItem")}
            </button>
          </div>

          {items.map((item) => (
            <div
              key={item.id}
              className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-3 sm:gap-6 mb-6 items-center"
            >
              <div className="relative">
                <input
                  type="text"
                  id="item-name"
                  className={`user-input peer`}
                  placeholder=" "
                  value={item.name}
                  onChange={(e) =>
                    handleChange(item.id, "name", e.target.value)
                  }
                />
                <label
                  htmlFor="item-name"
                  className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                >
                  {t("label.itemName")}{" "}
                  <span className="text-merchant-error">*</span>
                </label>
              </div>

              <div className="relative">
                <input
                  type="number"
                  id="quantity"
                  className={`user-input peer`}
                  placeholder=" "
                  value={item.quantity}
                  onChange={(e) =>
                    handleChange(item.id, "quantity", e.target.value)
                  }
                />
                <label
                  htmlFor="quantity"
                  className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                >
                  {t("label.quantity")}{" "}
                  <span className="text-merchant-error">*</span>
                </label>
              </div>

              <div className="relative">
                <input
                  type="number"
                  id="unit-price"
                  className={`user-input peer`}
                  placeholder=" "
                  value={item.price}
                  onChange={(e) =>
                    handleChange(item.id, "price", e.target.value)
                  }
                />
                <label
                  htmlFor="unit-price"
                  className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                >
                  {t("label.unitPrice")}{" "}
                  <span className="text-merchant-error">*</span>
                </label>
              </div>

              <div className="relative">
                <input
                  type="number"
                  id="subtotal"
                  className={`user-input peer`}
                  placeholder=" "
                  value={item.subtotal}
                  readOnly
                />
                <label
                  htmlFor="subtotal"
                  className="user-label absolute -translate-y-4 scale-75 top-2 z-10 origin-[0] px-[0px] peer-focus:px-2 peer-focus:text-fg-brand peer-placeholder-shown:scale-100 peer-placeholder-shown:-translate-y-1/2 peer-placeholder-shown:top-1/2 peer-focus:top-2 peer-focus:scale-[90%] peer-focus:-translate-y-4  start-[16px]"
                >
                  {t("label.subTotal")}{" "}
                  <span className="text-merchant-error">*</span>
                </label>
              </div>

              <div>
                <button
                  type="button"
                  onClick={() => handleRemoveItem(item.id)}
                  className="h-[48px] flex items-center justify-center gap-[8px] w-full bg-merchant-error hover:bg-red-700 text-white rounded-[8px] font-medium transition-all duration-200 ease-in-out cursor-pointer"
                >
                  {t("buttonText.remove")}
                </button>
              </div>
            </div>
          ))}
          <div className="flex justify-start">
            <button
              className="group primary-button w-full"
              type="submit"
              disabled={loading}
            >
              <span className="primary-button-hover-effect"></span>
              <span className="primary-button-text">
                {loading ? t("buttonText.updating") : t("buttonText.update")}
              </span>
            </button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default EditInvoice;
