"use client";

import logo from "@/assets/logo/logo.png";
import { useSettings } from "@/context/settingsContext";
import ApiPath from "@/network/api/apiPath";
import NetworkService from "@/network/service/networkService";
import { dynamicDecimals, getSettingValue } from "@/utils/utils";
import { Icon } from "@iconify/react";
import Image from "next/image";
import Link from "next/link";
import { useEffect, useState } from "react";

const InvoiceDetails = ({ invoiceId }) => {
  const network = new NetworkService();
  const [singleInvoiceData, setSingleInvoiceData] = useState({});
  const [loading, setLoading] = useState(true);
  const { settings } = useSettings();
  const siteCurrencyCode = getSettingValue(settings, "site_currency");
  const siteCurrencyDecimals = getSettingValue(
    settings,
    "site_currency_decimals"
  );

  // fetch invoice data
  const fetchSingleInvoiceData = async () => {
    try {
      setLoading(true);
      const res = await network.get(ApiPath.invoiceID(invoiceId));
      setSingleInvoiceData(res.data.data);
    } finally {
      setLoading(false);
    }
  };

  // handle print functionality
  const handlePrint = () => {
    window.print();
  };

  const SkeletonCard = () => (
    <div className="animate-pulse max-w-[900px] mx-auto bg-merchant-bg-one rounded-[16px] p-5 sm:p-8">
      <div className="flex flex-col sm:flex-row justify-between items-start mb-8 gap-4">
        <div className="h-[50px] w-[120px] bg-gray-200 rounded"></div>
        <div className="space-y-2 w-[180px]">
          <div className="h-3 bg-gray-200 rounded w-full"></div>
          <div className="h-3 bg-gray-200 rounded w-3/4"></div>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mb-8">
        <div className="space-y-3">
          <div className="h-4 bg-gray-200 rounded w-2/3"></div>
          <div className="h-3 bg-gray-200 rounded w-1/2"></div>
          <div className="h-3 bg-gray-200 rounded w-3/4"></div>
          <div className="h-5 bg-gray-200 rounded w-[80px]"></div>
        </div>
        <div className="space-y-3 text-right">
          <div className="h-4 bg-gray-200 rounded w-[100px] ms-auto"></div>
          <div className="h-6 bg-gray-200 rounded w-[120px] ms-auto"></div>
          <div className="h-3 bg-gray-200 rounded w-[140px] ms-auto"></div>
        </div>
      </div>

      <div className="h-5 bg-gray-200 rounded w-1/3 mb-4"></div>
      <div className="w-full border-t border-gray-200 mb-3"></div>

      <div className="space-y-3">
        {[1, 2, 3].map((i) => (
          <div key={i} className="flex justify-between text-gray-500 text-sm">
            <div className="h-3 bg-gray-200 rounded w-1/4"></div>
            <div className="h-3 bg-gray-200 rounded w-1/6"></div>
          </div>
        ))}
      </div>

      <div className="mt-8 flex justify-end gap-4">
        <div className="h-[44px] w-[120px] bg-gray-200 rounded"></div>
        <div className="h-[44px] w-[120px] bg-gray-200 rounded"></div>
      </div>
    </div>
  );

  useEffect(() => {
    fetchSingleInvoiceData();
  }, []);

  return (
    <div>
      <Link
        href="/dashboard/invoice"
        className="text-merchant-text text-[14px] font-bold flex items-center gap-2 hover:underline hover:text-merchant-primary transition-all duration-300 ease-in-out mb-3"
      >
        <Icon
          icon="lucide:arrow-left"
          width="20"
          height="20"
          className="w-[20px] h-[20px]"
        />
        Back
      </Link>
      <div className="printable-invoice-details">
        {loading ? (
          <SkeletonCard />
        ) : (
          <div className="max-w-[900px] mx-auto bg-merchant-bg-one rounded-[16px] p-5 sm:p-8 invoice-full">
            <div className="flex flex-col sm:flex-row gap-2 justify-between items-start mb-8">
              <div className="flex items-center gap-3">
                <div className="logo h-[30px] sm:h-[40px] w-auto">
                  <Image
                    src={logo}
                    alt="Logo"
                    width={100}
                    height={100}
                    className="h-full w-auto"
                  />
                </div>
              </div>
              <div className="text-left sm:text-right text-gray-700">
                <div className="text-sm">Ref: #{singleInvoiceData.number}</div>
                <div className="text-sm">
                  Issued:
                  {new Date(singleInvoiceData.issue_date).toLocaleDateString(
                    "en-US",
                    {
                      day: "2-digit",
                      month: "short",
                      year: "numeric",
                    }
                  )}
                </div>
              </div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mb-8">
              <div>
                <h2 className="text-xl font-bold text-merchant-text mb-4">
                  {singleInvoiceData.to}
                </h2>
                <div className="text-gray-700 mb-2">
                  {singleInvoiceData.email}
                </div>
                <div className="text-gray-700 mb-4">
                  {singleInvoiceData.address}
                </div>
                <div className="inline-block">
                  {singleInvoiceData.is_paid ? (
                    <span className="badge badge-success">Paid</span>
                  ) : (
                    <span className="badge badge-error">Unpaid</span>
                  )}
                </div>
              </div>

              <div className="text-left md:text-right">
                <h3 className="text-lg font-bold text-merchant-text mb-2">
                  Total Amount:
                </h3>
                <div className="text-xl sm:text-2xl font-bold text-merchant-text mb-4">
                  {Number(singleInvoiceData.total_amount).toFixed(
                    dynamicDecimals({
                      currencyCode: singleInvoiceData.currency,
                      siteCurrencyCode: siteCurrencyCode,
                      siteCurrencyDecimals: siteCurrencyDecimals,
                      isCrypto: singleInvoiceData.is_crypto,
                    })
                  )}{" "}
                  {singleInvoiceData.currency}
                </div>
                <div className="text-merchant-text font-medium text-sm space-y-1">
                  <div>
                    Amount:{" "}
                    {Number(singleInvoiceData.amount).toFixed(
                      dynamicDecimals({
                        currencyCode: singleInvoiceData.currency,
                        siteCurrencyCode: siteCurrencyCode,
                        siteCurrencyDecimals: siteCurrencyDecimals,
                        isCrypto: singleInvoiceData.is_crypto,
                      })
                    )}{" "}
                    {singleInvoiceData.currency}
                  </div>
                  <div>
                    Charge:{" "}
                    {Number(singleInvoiceData.charge).toFixed(
                      dynamicDecimals({
                        currencyCode: singleInvoiceData.currency,
                        siteCurrencyCode: siteCurrencyCode,
                        siteCurrencyDecimals: siteCurrencyDecimals,
                        isCrypto: singleInvoiceData.is_crypto,
                      })
                    )}{" "}
                    {singleInvoiceData.currency}
                  </div>
                </div>
              </div>
            </div>

            <div className="mb-8">
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b border-merchant-border">
                      <th className="text-left py-4 px-0 text-[16px] font-semibold text-merchant-text">
                        Item Name
                      </th>
                      <th className="text-left py-4 px-4 text-[16px] font-semibold text-merchant-text">
                        Quantity
                      </th>
                      <th className="text-left py-4 px-4 text-[16px] font-semibold text-merchant-text">
                        Unit Price
                      </th>
                      <th className="text-right py-4 px-0 text-[16px] font-semibold text-merchant-text">
                        Subtotal
                      </th>
                    </tr>
                  </thead>
                  {singleInvoiceData?.items?.map((item) => (
                    <tbody className="divide-y divide-gray-200" key={item.id}>
                      <tr className="text-merchant-text text-[16px] font-medium">
                        <td className="py-4 px-0">{item.name}</td>
                        <td className="py-4 px-4">{item.quantity}</td>
                        <td className="py-4 px-4">
                          {parseFloat(item.unit_price).toFixed(
                            dynamicDecimals({
                              currencyCode: singleInvoiceData.currency,
                              siteCurrencyCode: siteCurrencyCode,
                              siteCurrencyDecimals: siteCurrencyDecimals,
                              isCrypto: singleInvoiceData.is_crypto,
                            })
                          )}{" "}
                          {singleInvoiceData.currency}
                        </td>
                        <td className="py-4 px-0 text-right">
                          {parseFloat(item.subtotal).toFixed(
                            dynamicDecimals({
                              currencyCode: singleInvoiceData.currency,
                              siteCurrencyCode: siteCurrencyCode,
                              siteCurrencyDecimals: siteCurrencyDecimals,
                              isCrypto: singleInvoiceData.is_crypto,
                            })
                          )}{" "}
                          {singleInvoiceData.currency}
                        </td>
                      </tr>
                    </tbody>
                  ))}
                </table>
              </div>

              <div className="mt-6 border-t border-merchant-border pt-[10px]">
                <div className="space-y-2">
                  <div className="flex justify-between items-center text-gray-600 font-medium text-[16px]">
                    <span>Subtotal</span>
                    <span>
                      {Number(singleInvoiceData.amount).toFixed(
                        dynamicDecimals({
                          currencyCode: singleInvoiceData.currency,
                          siteCurrencyCode: siteCurrencyCode,
                          siteCurrencyDecimals: siteCurrencyDecimals,
                          isCrypto: singleInvoiceData.is_crypto,
                        })
                      )}{" "}
                      {singleInvoiceData.currency}
                    </span>
                  </div>
                  <div className="flex justify-between items-center text-gray-600 font-medium text-[16px]">
                    <span>Charge</span>
                    <span>
                      {Number(singleInvoiceData.charge).toFixed(
                        dynamicDecimals({
                          currencyCode: singleInvoiceData.currency,
                          siteCurrencyCode: siteCurrencyCode,
                          siteCurrencyDecimals: siteCurrencyDecimals,
                          isCrypto: singleInvoiceData.is_crypto,
                        })
                      )}{" "}
                      {singleInvoiceData.currency}
                    </span>
                  </div>
                  <div className="flex justify-between items-center text-lg sm:text-[20px] text-merchant-text font-bold border-t border-merchant-border pt-2">
                    <span className="">Total Amount</span>
                    <span className="text-merchant-primary">
                      {Number(singleInvoiceData.total_amount).toFixed(
                        dynamicDecimals({
                          currencyCode: singleInvoiceData.currency,
                          siteCurrencyCode: siteCurrencyCode,
                          siteCurrencyDecimals: siteCurrencyDecimals,
                          isCrypto: singleInvoiceData.is_crypto,
                        })
                      )}{" "}
                      {singleInvoiceData.currency}
                    </span>
                  </div>
                </div>
              </div>
            </div>

            <div className="flex gap-4 justify-end">
              <button
                className="group primary-button primary-border primary-button-xl"
                onClick={handlePrint}
              >
                <span className="primary-button-hover-effect"></span>
                <span className="primary-button-text">Print Invoice</span>
              </button>
              {!singleInvoiceData?.is_paid && (
                <a
                  href={
                    singleInvoiceData?.transaction?.payment_gateway_url +
                    "?redirect_url=" +
                    window.location.href
                  }
                  rel="noopener noreferrer"
                  className="group primary-button primary-button-xl"
                >
                  <span className="primary-button-hover-effect"></span>
                  <span className="primary-button-text">Pay Now</span>
                </a>
              )}
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default InvoiceDetails;
