"use client";
import Badge from "@/components/common/Badge";
import LoadingSpinner from "@/components/common/LoadingSpinner";
import NoDataFound from "@/components/common/NoDataFound";
import Pagination from "@/components/common/Pagination";
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import { useSettings } from "@/context/settingsContext";
import { useT } from "@/context/TranslationContext";
import ApiPath from "@/network/api/apiPath";
import NetworkService from "@/network/service/networkService";
import {
  CreatedAtFormatDate,
  dynamicDecimals,
  formatLocalDate,
  getSettingValue,
} from "@/utils/utils";
import { Icon } from "@iconify/react";
import Link from "next/link";
import { useEffect, useState } from "react";
import Flatpickr from "react-flatpickr";

const InvoiceHistory = () => {
  const network = new NetworkService();
  const [invoiceHistory, setInvoiceHistory] = useState([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [perPage, setPerPage] = useState(15);
  const [lastPage, setLastPage] = useState(1);
  const [total, setTotal] = useState(0);
  const [loading, setLoading] = useState(false);
  const [hasFetched, setHasFetched] = useState(false);
  const [status, setStatus] = useState("");
  const [startDate, setStartDate] = useState("");
  const [endDate, setEndDate] = useState("");
  const { settings } = useSettings();
  const siteCurrency = getSettingValue(settings, "site_currency");
  const siteCurrencyDecimals = getSettingValue(
    settings,
    "site_currency_decimals"
  );
  const t = useT();

  // get invoice history data
  const getInvoiceHistory = async () => {
    setLoading(true);
    try {
      const res = await network.get(ApiPath.createInvoice, {
        page: currentPage,
        per_page: perPage,
        status: status,
        date_from: startDate,
        date_to: endDate,
      });
      setInvoiceHistory(res.data.data.invoices);
      setPerPage(res.data.data.pagination.per_page);
      setLastPage(res.data.data.pagination.last_page);
      setTotal(res.data.data.pagination.total);
    } finally {
      setLoading(false);
      setHasFetched(true);
    }
  };

  // page change
  const handlePageClick = (event) => {
    const selectedPage = event.selected + 1;
    setCurrentPage(selectedPage);
  };

  useEffect(() => {
    getInvoiceHistory();
  }, [currentPage, status, startDate, endDate]);

  return (
    <>
      <div>
        <div className="mb-[30px]">
          <h4 className="text-[20px] font-bold text-merchant-text">
            {t("title.invoiceHistory")}
          </h4>
        </div>
        <div className="border border-[rgba(26,32,44,0.10)] p-[20px] rounded-[8px]">
          <div className="filter flex flex-wrap items-center justify-between gap-[16px] mb-[30px]">
            <div className="w-full sm:w-lg">
              <div className="flex flex-col sm:flex-row items-start sm:items-center gap-2">
                <div className="w-full sm:w-max">
                  <Flatpickr
                    className="border border-merchant-input-border font-bold rounded-[8px] px-3 h-[40px] text-sm focus:outline-none focus:ring-1 focus:ring-transparent focus:border-merchant-primary w-full sm:w-max"
                    placeholder={t("placeholder.selectDateRange")}
                    options={{
                      mode: "range",
                      dateFormat: "Y-m-d",
                      closeOnSelect: false,
                    }}
                    onChange={(selectedDates) => {
                      if (selectedDates.length === 2) {
                        const [start, end] = selectedDates;
                        setStartDate(formatLocalDate(start));
                        setEndDate(formatLocalDate(end));
                      }
                    }}
                  />
                </div>
                <div className="w-full sm:w-sm">
                  <Select
                    value={status}
                    onValueChange={(value) => setStatus(value)}
                  >
                    <SelectTrigger className="w-full !h-[40px] border border-merchant-input-border rounded-[8px] text-merchant-text !text-[14px]">
                      <SelectValue
                        placeholder={t("placeholder.selectStatus")}
                      />
                    </SelectTrigger>

                    <SelectContent className="rounded-[8px]">
                      <SelectGroup>
                        <SelectItem value="paid">Paid</SelectItem>
                        <SelectItem value="unpaid">Unpaid</SelectItem>
                        <SelectItem value="published">Published</SelectItem>
                        <SelectItem value="draft">Draft</SelectItem>
                      </SelectGroup>
                    </SelectContent>
                  </Select>
                </div>
                <button className="group primary-button primary-button-md w-full sm:w-1/2">
                  <span className="primary-button-hover-effect"></span>
                  <span className="primary-button-text">
                    {t("buttonText.search")}
                  </span>
                </button>
              </div>
            </div>
            <div>
              <Link
                href="/dashboard/invoice/create-invoice"
                className="group primary-button primary-button-md"
              >
                <span className="primary-button-hover-effect"></span>
                <span className="primary-button-text">
                  {t("buttonText.createInvoice")}
                </span>
              </Link>
            </div>
          </div>
          <div className="bg-white rounded-[8px]">
            <div className="max-w-full overflow-x-auto">
              <div className="common-table min-w-[1020px]">
                <div className="full-table">
                  <table className="w-full whitespace-nowrap">
                    <thead className="table-head">
                      <tr className="table-head-tr">
                        <th className="table-th">{t("tableHead.invoiceTo")}</th>
                        <th className="table-th">
                          {t("tableHead.emailAddress")}
                        </th>
                        <th className="table-th">{t("tableHead.amount")}</th>
                        <th className="table-th">{t("tableHead.charge")}</th>
                        <th className="table-th">{t("tableHead.payment")}</th>
                        <th className="table-th">{t("tableHead.status")}</th>
                        <th className="table-th">{t("tableHead.action")}</th>
                      </tr>
                    </thead>

                    <tbody className="table-body">
                      {loading ? (
                        <tr>
                          <td colSpan="7" className="text-center pt-5">
                            <div className="flex justify-center items-center">
                              <LoadingSpinner />
                            </div>
                          </td>
                        </tr>
                      ) : hasFetched && invoiceHistory.length === 0 ? (
                        <tr>
                          <td colSpan="7" className="py-[40px] text-center">
                            <NoDataFound message="No Invoice Data Found" />
                          </td>
                        </tr>
                      ) : (
                        <>
                          {invoiceHistory.map((invoice, index) => (
                            <tr className="table-body-tr last:!border-0">
                              <td className="table-td">
                                <div className="flex items-center gap-[10px]">
                                  <div>
                                    <h6 className="text-[13px] font-bold text-merchant-text mb-[5px]">
                                      {invoice?.to}
                                    </h6>
                                    <p className="text-[12px] text-merchant-paragraph font-semibold">
                                      <CreatedAtFormatDate
                                        createdAt={invoice?.created_at}
                                      />
                                    </p>
                                  </div>
                                </div>
                              </td>
                              <td className="table-td">
                                <span className="text-[13px] text-merchant-text font-semibold">
                                  {invoice?.email}
                                </span>
                              </td>
                              <td className="table-td">
                                <div className="flex items-center gap-[5px]">
                                  <Icon
                                    icon="lucide:arrow-up"
                                    width="20"
                                    height="20"
                                    className={`w-[20px] h-[20px] text-merchant-success`}
                                  />
                                  <span className="block text-[13px] text-merchant-text font-semibold">
                                    +
                                    {Number(invoice?.total_amount).toFixed(
                                      dynamicDecimals({
                                        currencyCode: invoice?.currency,
                                        siteCurrencyCode: siteCurrency,
                                        siteCurrencyDecimals:
                                          siteCurrencyDecimals,
                                        isCrypto: invoice?.is_crypto,
                                      })
                                    )}{" "}
                                    {invoice.currency}
                                  </span>
                                </div>
                              </td>
                              <td className="table-td">
                                <div className="flex items-center gap-[5px]">
                                  <Icon
                                    icon="lucide:arrow-down"
                                    width="20"
                                    height="20"
                                    className="w-[20px] h-[20px] text-merchant-error"
                                  />
                                  <span className="block text-[13px] text-merchant-text font-semibold">
                                    -
                                    {Number(invoice?.charge).toFixed(
                                      dynamicDecimals({
                                        currencyCode: invoice?.currency,
                                        siteCurrencyCode: siteCurrency,
                                        siteCurrencyDecimals:
                                          siteCurrencyDecimals,
                                        isCrypto: invoice?.is_crypto,
                                      })
                                    )}{" "}
                                    {invoice.currency}
                                  </span>
                                </div>
                              </td>
                              <td className="table-td">
                                {invoice?.is_paid ? (
                                  <span className="badge-paid">Paid</span>
                                ) : (
                                  <span className="badge-unpaid">Unpaid</span>
                                )}
                              </td>
                              <td className="table-td">
                                {invoice?.is_published ? (
                                  <Badge status="published" />
                                ) : (
                                  <Badge status="draft" />
                                )}
                              </td>
                              <td className="table-td">
                                <div className="flex items-center gap-[18px]">
                                  {!invoice?.is_paid && (
                                    <Link
                                      href={`/dashboard/invoice/${invoice?.id}/invoice-edit`}
                                      className="flex justify-center items-center"
                                    >
                                      <Tooltip>
                                        <TooltipTrigger className="cursor-pointer">
                                          <Icon
                                            icon="lucide:edit"
                                            width="20"
                                            height="20"
                                            className="w-[20px] h-[20px] text-user-text"
                                          />
                                        </TooltipTrigger>
                                        <TooltipContent>
                                          <p>{t("buttonText.editInvoice")}</p>
                                        </TooltipContent>
                                      </Tooltip>
                                    </Link>
                                  )}

                                  <Link
                                    href={`/dashboard/invoice/${invoice?.id}/invoice-details`}
                                    className="view-details-button"
                                  >
                                    {t("buttonText.view")}
                                  </Link>
                                </div>
                              </td>
                            </tr>
                          ))}
                        </>
                      )}
                    </tbody>
                  </table>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      {invoiceHistory.length > 0 && (
        <div className="mt-[30px]">
          <Pagination
            lastPage={lastPage}
            handlePageClick={handlePageClick}
            currentPage={currentPage}
            perPage={perPage}
            total={total}
          />
        </div>
      )}
    </>
  );
};

export default InvoiceHistory;
